function [period,phase_info]=fourier_analysis(D,object)

% Performing the fast Fourier transform (FFT) on neurons (or genes) using the fft function.
% FFT maps a function in a time domain into a frequency domain such that sharp 
% peaks appear at frequencies where there is intrinsic periodicity. 
% The transformed information including power spectrum and corresponding phase are calculated. 
% The frequency of maximum power is deemed the major frequency and used in determining the phase information.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Fourier analysis of input matrix D
f=(0:size(D,2)-1)'/size(D,2);% frequency 
T=[Inf;1./f(2:end)];% period 

phase_info=zeros(size(D,1),1);
period=zeros(size(D,1),1);

for i=1:length(D)
    % Discrete Fourier transform computed with fast fourier transform algorithm
    % converting to the frequency domain from time domain
    if strcmp(object,'neuron')
        y=fft(D(i,:)); % FFT on neuron
    elseif strcmp(object,'gene')
        % loess smoothing is performed to impute missing values
        D_smoothed=smooth(0:size(D,2)-1,D(i,:),0.12,'loess');
        y=fft(D_smoothed);% FFT on smoothed gene
    end
    % The power spectrum, a measurement of the power at various frequencies 
    pyy=y.*conj(y)/size(D,2);
    % The frequency of maximum power is deemed the major frequency and used in determining the phase information
    [mp,index] = max(pyy);% index for maximum power
    I=imag(y(index));% the imaginary part of the discrete Fourier transform
    R=real(y(index));% the real part of the discrete Fourier transform    
    phi=atan2(-I,R); % phase information lying in the closed interval [-pi,pi]
    
    % Save the infomation about the period and corresponding phase
    period(i,1)=T(index);
    phase_info(i,1)=phi;    
end

return